/* pam_userdb module */
 
/*
 * $Id: pam_userdb.c,v 1.1.1.2 2004/06/22 22:45:52 dlehman Exp $
 * Written by Cristian Gafton <gafton@redhat.com> 1996/09/10
 * See the end of the file for Copyright Information
 */

#include "../../_pam_aconf.h"

#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <syslog.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>

#ifdef HAVE_LIBDB
# define DB_DBM_HSEARCH    1 /* use the dbm interface */
# if defined(__GLIBC__) && ((__GLIBC__ > 2) || ((__GLIBC__ == 2) && (__GLIBC_MINOR__ > 1)))
#  include <db3/db.h>
# else
#  include <db.h>
# endif
#else
# ifdef HAVE_DB_H
#  define DB_DBM_HSEARCH    1 /* use the dbm interface */
#  include <db.h>
# else
#  error "failed to find a libdb or equivalent"
# endif
#endif

#include "pam_userdb.h"

/*
 * here, we make a definition for the externally accessible function
 * in this file (this definition is required for static a module
 * but strongly encouraged generally) it is used to instruct the
 * modules include file to define the function prototypes.
 */

#define PAM_SM_AUTH
#define PAM_SM_ACCOUNT

#include "../../libpam/include/security/pam_modules.h"

/* some syslogging */

static void _pam_log(int err, const char *format, ...)
{
    va_list args;

    va_start(args, format);
    openlog(MODULE_NAME, LOG_CONS|LOG_PID, LOG_AUTH);
    vsyslog(err, format, args);
    va_end(args);
    closelog();
}

char * database	 = NULL;
static int ctrl	 = 0;

static int _pam_parse(int argc, const char **argv)
{
     /* step through arguments */
     for (ctrl = 0; argc-- > 0; ++argv) {

          /* generic options */

          if (!strcmp(*argv,"debug"))
               ctrl |= PAM_DEBUG_ARG;
	  else if (!strcasecmp(*argv, "icase"))
	      ctrl |= PAM_ICASE_ARG;
	  else if (!strcasecmp(*argv, "dump"))
	      ctrl |= PAM_DUMP_ARG;
	  else if (!strcasecmp(*argv, "use_authtok"))
	      ctrl |= PAM_USE_AUTHTOK_ARG;
	  else if (!strcasecmp(*argv, "unknown_ok"))
	      ctrl |= PAM_UNKNOWN_OK_ARG;
          else if (!strncasecmp(*argv,"db=", 3)) {
	      database = strdup((*argv) + 3);
	      if (database == NULL)
		  _pam_log(LOG_ERR, "pam_parse: could not parse argument \"%s\"",
			   *argv);
	  }
	  else if (!strcasecmp(*argv,"key_only")) {
	    ctrl |= PAM_KEY_ONLY;
	  } else {
               _pam_log(LOG_ERR, "pam_parse: unknown option; %s", *argv);
          }
     }

     return ctrl;
}


/*
 * Looks up an user name in a database and checks the password
 *
 * return values:
 *	 1  = User not found
 *	 0  = OK
 * 	-1  = Password incorrect
 *	-2  = System error
 */
static int user_lookup(const char *user, const char *pass)
{
    DBM *dbm;
    datum key, data;
    char *key_only_buf = NULL;
    int key_only_len = 0;

    if (ctrl & PAM_KEY_ONLY) {
      /* set up a buffer for the username and password pair */
      key_only_len = strlen(user) + 1 + strlen(pass);
      key_only_buf = alloca(key_only_len + 1);
      /* check to make sure that we have allocated the memory */
      if (key_only_buf == NULL) {
	_pam_log(LOG_CRIT, "no memory for key_only");
	return -2;
      }
      /* add the user */
      strcpy(key_only_buf, user);
      /* add the "-" */
      strcat(key_only_buf, "-");
      /* add the password */
      strcat(key_only_buf, pass);
    }

    /* Open the DB file. */
    dbm = dbm_open(database, O_RDONLY, 0644);
    if (dbm == NULL) {
	_pam_log(LOG_ERR, "%s: could not open database `%s'", __FUNCTION__, database);
	return -2;
    }

    if (ctrl &PAM_DUMP_ARG) {
	_pam_log(LOG_INFO, "Database dump:");
	for (key = dbm_firstkey(dbm);  key.dptr != NULL; key = dbm_nextkey(dbm)) {
	    data = dbm_fetch(dbm, key);
	    _pam_log(LOG_INFO, "key[len=%d] = `%s', data[len=%d] = `%s'",
		     key.dsize, key.dptr, data.dsize, data.dptr);
	}
    } 
    /* do some more init work */
    memset(&key, 0, sizeof(key));
    memset(&data, 0, sizeof(data));

    /* use the key_only_buf if it exists. */
    if (key_only_buf) {
      (const char *) key.dptr = key_only_buf;
      key.dsize = key_only_len;
    } else {
      (const char *) key.dptr = user;
      key.dsize = strlen(user);
    }
    
    data = dbm_fetch(dbm, key);  
    if (ctrl & PAM_DEBUG_ARG) {
	_pam_log(LOG_INFO, "password in database is [%p] `%s', len is %d",
		 data.dptr, (char *) data.dptr, data.dsize);
    }
    if (data.dptr != NULL) {
	int compare = 0;
	/* bingo, got it */
	if (ctrl & PAM_KEY_ONLY) {
	  /* if we're in key only mode then matching
	     the key is enough to return success */
	  compare = 0;
	} else if (ctrl & PAM_ICASE_ARG) {
	    compare = strncasecmp(pass, data.dptr, data.dsize);
	} else {
	    compare = strncmp(pass, data.dptr, data.dsize);
	}
	dbm_close(dbm);
	if (compare == 0)
	    return 0; /* match */
	else
	    return -1; /* wrong */
    } else {
	if (ctrl & PAM_DEBUG_ARG) {    
	    _pam_log(LOG_INFO, "error returned by dbm_fetch: %s", strerror(errno));
	}
	dbm_close(dbm);
	/* probably we should check dbm_error() here */
	return 1; /* not found */
    }
	    
    if (dbm != NULL)
	dbm_close(dbm);
    /* we don't know what the heck happend */
    return -2;
}

/*
 * Checks if a user has an entry in the database
 *
 * return values:
 *	 1  = User not found
 *	 0  = OK
 * 	-1  = System error
 */
static int valid_user(const char *user)
{
    DBM *dbm;
    datum key, data;

    /* Open the DB file. */
    dbm = dbm_open(database, O_RDONLY, 0644);
    if (dbm == NULL) {
	_pam_log(LOG_ERR, "%s: could not open database `%s'", __FUNCTION__, database);
	return -1;
    }

    if (ctrl &PAM_DUMP_ARG) {
	_pam_log(LOG_INFO, "Database dump:");
	for (key = dbm_firstkey(dbm);  key.dptr != NULL; key = dbm_nextkey(dbm)) {
	    data = dbm_fetch(dbm, key);
	    _pam_log(LOG_INFO, "key[len=%d] = `%s', data[len=%d] = `%s'",
		     key.dsize, key.dptr, data.dsize, data.dptr);
	}
    } 
    /* do some more init work */
    memset(&key, 0, sizeof(key));
    memset(&data, 0, sizeof(data));
    (const char *) key.dptr = user;
    key.dsize = strlen(user);
    
    data = dbm_fetch(dbm, key);  

    if (data.dptr != NULL) {
	dbm_close(dbm);
	return 0;
    }

    if (ctrl & PAM_DEBUG_ARG) {    
	_pam_log(LOG_INFO, "error returned by dbm_fetch: %s", strerror(errno));
    }
    if (dbm != NULL)
	dbm_close(dbm);
    return 1; /* not found */
}

/* --- authentication management functions --- */

PAM_EXTERN
int pam_sm_authenticate(pam_handle_t *pamh, int flags,
			int argc, const char **argv)
{
     const char *username;
     const char *password;
     int retval = PAM_AUTH_ERR;
    
     /* parse arguments */
     ctrl = _pam_parse(argc, argv);

     /* Do we have a database ? */
     if (database == NULL) {
	 _pam_log(LOG_ERR, "no database supplied; must use db=<database_path> arg");
	 return PAM_SERVICE_ERR;
     }
     
     /* Get the username */
     retval = pam_get_user(pamh, &username, NULL);
     if ((retval != PAM_SUCCESS) || (!username)) {
        if (ctrl & PAM_DEBUG_ARG)
            _pam_log(LOG_DEBUG,"can not get the username");
        return PAM_SERVICE_ERR;
     }
     
     if ((ctrl & PAM_USE_AUTHTOK_ARG) == 0) {
	 /* Converse just to be sure we have the password */
	 retval = conversation(pamh);
	 if (retval != PAM_SUCCESS) {
	     _pam_log(LOG_ERR, "could not obtain password for `%s'",
		      username);
	     return PAM_CONV_ERR;
	 }
     }
     
     /* Get the password */
     retval = pam_get_item(pamh, PAM_AUTHTOK, (const void **) &password);
     if (retval != PAM_SUCCESS) {
	 _pam_log(LOG_ERR, "Could not retrive user's password");
	 return PAM_AUTHTOK_ERR;
     }
     
     if (ctrl & PAM_DEBUG_ARG)
	 _pam_log(LOG_INFO, "Verify user `%s' with password `%s'",
		  username, password);
     
     /* Now use the username to look up password in the database file */
     retval = user_lookup(username, password);
     switch (retval) {
	 case -2:
	     /* some sort of system error. The log was already printed */
	     return PAM_SERVICE_ERR;    
	 case -1:
	     /* incorrect password */
	     _pam_log(LOG_WARNING, "user `%s' denied access (incorrect password)", username);
	     return PAM_AUTH_ERR;
	 case 1:
	     /* the user does not exist in the database */
	     if (ctrl & PAM_DEBUG_ARG)
		 _pam_log(LOG_NOTICE, "user `%s' not found in %s",
			  username, database);
	     if (ctrl & PAM_UNKNOWN_OK_ARG) {
		 return PAM_IGNORE;
	     }	 
	     return PAM_USER_UNKNOWN;
	 case 0:
	     /* Otherwise, the authentication looked good */
	     _pam_log(LOG_NOTICE, "user '%s' granted acces", username);
	     return PAM_SUCCESS;
	 default:
	     /* we don't know anything about this return value */
	     _pam_log(LOG_ERR, "internal module error (retval = %d, user = `%s'",
		      retval, username);
	     return PAM_SERVICE_ERR;
     }
     /* should not be reached */
     return PAM_IGNORE;
}

PAM_EXTERN
int pam_sm_setcred(pam_handle_t *pamh, int flags,
		   int argc, const char **argv)
{
    return PAM_SUCCESS;
}

/*
 * Account management. We want to verify that the account exists in the given
 * database before returning PAM_SUCCESS
 */
PAM_EXTERN
int pam_sm_acct_mgmt(pam_handle_t *pamh, int flags,
		   int argc, const char **argv)
{
    const char *username;
    int retval = PAM_USER_UNKNOWN;

    /* parse arguments */
    ctrl = _pam_parse(argc, argv);

    /* Do we have a database ? */
    if (database == NULL) {
	_pam_log(LOG_ERR, "no database supplied; must use db=<database_path> arg");
	return PAM_SERVICE_ERR;
    }	

    /* Get the username */
    retval = pam_get_user(pamh, &username, NULL);
    if ((retval != PAM_SUCCESS) || (!username)) {
	if (ctrl & PAM_DEBUG_ARG)
	    _pam_log(LOG_DEBUG,"can not get the username");
	return PAM_SERVICE_ERR;
    }

    /* Verify the username */
    retval = valid_user(username);
    switch (retval) {
	case -1:
	    /* some sort of system error. The log was already printed */
	    return PAM_SERVICE_ERR;
	case 1:
	    /* the user does not exist in the database */
	    if (ctrl & PAM_DEBUG_ARG)
		_pam_log(LOG_NOTICE, "user `%s' not found in the database",
			 username);
	    if (ctrl & PAM_UNKNOWN_OK_ARG)
		return PAM_IGNORE;
	    return PAM_USER_UNKNOWN;
	case 0:
	    /* Otherwise, the authentication looked good */
	    _pam_log(LOG_NOTICE, "user '%s' granted acces", username);
	    return PAM_SUCCESS;
	default:
	    /* we don't know anything about this return value */
	    _pam_log(LOG_ERR, "internal module error (retval = %d, user = `%s'",
		     retval, username);
	    return PAM_SERVICE_ERR;
    }
    
    /* should not be reached */
    return PAM_IGNORE;
}


#ifdef PAM_STATIC

/* static module data */

struct pam_module _pam_userdb_modstruct = {
     MODULE_NAME,
     pam_sm_authenticate,
     pam_sm_setcred,
     pam_sm_acct_mgmt,
     NULL,
     NULL,
     NULL,
};

#endif

/*
 * Copyright (c) Cristian Gafton <gafton@redhat.com>, 1999
 *                                              All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, and the entire permission notice in its entirety,
 *    including the disclaimer of warranties.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote
 *    products derived from this software without specific prior
 *    written permission.
 *
 * ALTERNATIVELY, this product may be distributed under the terms of
 * the GNU Public License, in which case the provisions of the GPL are
 * required INSTEAD OF the above restrictions.  (This clause is
 * necessary due to a potential bad interaction between the GPL and
 * the restrictions contained in a BSD-style copyright.)
 *
 * THIS SOFTWARE IS PROVIDED `AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 */
