/*******************************************************************************
 * Copyright (c) 2001, 2017 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.rsadapter.impl;

import com.ibm.ws.rsadapter.AdapterUtil;

/**
 * A key used for PreparedStatement caching.
 */
public class PSCacheKey extends StatementCacheKey {
    public static final int PREPARED_STATEMENT = 1;

    /**
     * Create a new key for PreparedStatement caching.
     * 
     * @param theSQL the SQL for the PreparedStatement.
     * @param theType the ResultSet type.
     * @param theConcurrency the ResultSet concurrency.
     * @param theHoldability the cursor holdability
     * @param theAutoGeneratedKeys the autoGeneratedKeys flag
     * @param isoLev the isolation level associated with the statement (relevent for DB2 and cloudscape NS)
     * @param schema The schema associated with the connection that created this statement
     */
    public PSCacheKey(String theSQL, int theType, int theConcurrency, int theHoldability, int theAutoGeneratedKeys, int isoLev, String dbSchema) { 
        type = theType;
        concurrency = theConcurrency;
        hCode = (sql = theSQL).hashCode();
        statementType = PREPARED_STATEMENT;
        holdability = theHoldability;
        autoGeneratedKeys = theAutoGeneratedKeys;
        statementIsoLevel = isoLev; 
        schema = dbSchema;
    }

    /**
     * @param keyToCheck the key to compare with this key.
     * 
     * @return true if this key is equal to the key provided, otherwise false.
     */
    @Override
    public final boolean equals(Object keyToCheck) {
        try {
            // Try to avoid the string.equals if we can. 
            
            if(keyToCheck == null)
                return false;

            StatementCacheKey k = (StatementCacheKey) keyToCheck;

            return (sql == k.sql || sql.equals(k.sql)) &&
                   statementType == k.statementType &&
                   holdability == k.holdability && 
                   autoGeneratedKeys == k.autoGeneratedKeys && 
                   type == k.type &&
                   concurrency == k.concurrency &&
                   statementIsoLevel == k.statementIsoLevel && // if db doesn't cache iso in statement then the value is 0 and it will be an extra 0=0
                   AdapterUtil.match(schema, k.schema);
        } catch (Exception ex) {
            // No FFDC code needed.
            return false;
        }
    }

    /**
     * @return a String representing this key.
     */
    @Override
    public final String toString() {
        StringBuffer sb = new StringBuffer("PSTMT: ").append(sql).append(' ').append(type).append(' ');
        sb.append(concurrency).append(' ').append(holdability).append(' ').append(autoGeneratedKeys).append(' ');
        sb.append(statementIsoLevel);
        if(schema != null)
            sb.append(' ').append(schema);
        return sb.toString();
    }
}
