/*******************************************************************************
 * Copyright (c) 2004, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.sib.comms.client;

import com.ibm.websphere.ras.TraceComponent;
import com.ibm.ejs.ras.TraceNLS;
import com.ibm.websphere.sib.exception.SIErrorException;
import com.ibm.websphere.sib.exception.SIIncorrectCallException;
import com.ibm.websphere.sib.exception.SIResourceException;
import com.ibm.ws.sib.comms.CommsConstants;
import com.ibm.ws.sib.comms.common.CommsByteBuffer;
import com.ibm.ws.sib.jfapchannel.Conversation;
import com.ibm.ws.sib.jfapchannel.JFapChannelConstants;
import com.ibm.ws.sib.utils.ras.SibTr;
import com.ibm.wsspi.sib.core.SIUncoordinatedTransaction;
import com.ibm.wsspi.sib.core.exception.SIConnectionDroppedException;
import com.ibm.wsspi.sib.core.exception.SIConnectionLostException;
import com.ibm.wsspi.sib.core.exception.SIRollbackException;

/**
 * This class is the proxy class for an uncoordinated (local) transaction.
 * <p>
 * Local transactions must be able to be used across multiple connections that
 * use the same physical link and connected to the same ME. As such, transaction Id's
 * must be unique across the physical link and need to be generated by the client.
 * This will result in the server keeping a table mapping the actual local transaction
 * object with the ID being used on the client.
 * <p>
 * This object needs to have strict synchronisation rules to prevent multiple
 * threads from calling these methods at the same time - as such all the methods
 * in this class are synchronised. However, other classes using this object should
 * obtain this instance's monitor before doing any work with it. They should also use
 * the <code>isValid()</code> method to determine whether the transaction is still
 * allowed to be used.
 * 
 * @author Gareth Matthews
 */
public class LocalTransactionProxy extends Transaction implements SIUncoordinatedTransaction
{
   /** Register Class with Trace Component */
   private static final TraceComponent tc = SibTr.register(LocalTransactionProxy.class, 
                                                           CommsConstants.MSG_GROUP, 
                                                           CommsConstants.MSG_BUNDLE);
      
   /** The NLS reference */
   private static final TraceNLS nls = TraceNLS.getTraceNLS(CommsConstants.MSG_BUNDLE);
   
   /** Flag to mark this transaction as invalid */
   private boolean valid = true;
   
   /** Log Source code level on static load of class */
   static 
   {
      if (tc.isDebugEnabled()) SibTr.debug(tc, "Source info: @(#)SIB/ws/code/sib.comms.client.impl/src/com/ibm/ws/sib/comms/client/LocalTransactionProxy.java, SIB.comms, WASX.SIB, uu1215.01 1.40");
   }

   /**
    * Contructor which assigns a transaction Id to this instance.
    * 
    * @param con
    * @param cp
    */
   public LocalTransactionProxy(Conversation con, ConnectionProxy cp)
   {
      super(con, cp);

      if (tc.isEntryEnabled()) SibTr.entry(this, tc, "<init>");
      if (tc.isEntryEnabled()) SibTr.exit(this, tc, "<init>");
   }
      
   /**
    * Commits this transaction by flowing the commit to the server and marking this 
    * transaction as invalid.
    */
   public synchronized void commit()
      throws SIIncorrectCallException, 
             SIRollbackException,
             SIResourceException,
             SIConnectionLostException, 
             SIErrorException
   {
      if (tc.isEntryEnabled()) SibTr.entry(this, tc, "commit");
      
      if (!valid) 
      {
         throw new SIIncorrectCallException(
            nls.getFormattedMessage("TRANSACTION_COMPLETE_SICO1022", null, null)
         );
      }
      
      // Mark this as invalid
      valid = false;
      
      CommsByteBuffer request = getCommsByteBuffer();
      request.putShort(getConnectionObjectID());
      request.putInt(getTransactionId());

      CommsByteBuffer reply = null;
      try
      {
         // Pass on call to server
         reply = jfapExchange(request,
                              JFapChannelConstants.SEG_COMMIT_TRANSACTION,
                              lowestPriority,
                              true);

         short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_COMMIT_TRANSACTION_R);
         if (err != CommsConstants.SI_NO_EXCEPTION)
         {
            checkFor_SIIncorrectCallException(reply, err);
            checkFor_SIRollbackException(reply, err);
            checkFor_SIResourceException(reply, err);
            checkFor_SIConnectionLostException(reply, err);
            checkFor_SIErrorException(reply, err);
            defaultChecker(reply, err);
         }
      }
      catch (SIConnectionDroppedException e)
      {
         // No FFDC Code needed
         // In this case we translate this to an SIConnectionLostException - as transactions aren't
         // tided to a particular connection
         throw new SIConnectionLostException(e.getMessage(), e);
      }
      finally
      {
         if (reply != null) reply.release();
      }
      
      if (tc.isEntryEnabled()) SibTr.exit(this, tc, "commit");
   }

   /**
    * Rolls back this transaction by flowing the rollback to the server and marking this 
    * transaction as invalid.
    * 
    * @throws com.ibm.websphere.sib.exception.SIIncorrectCallException
    * @throws com.ibm.wsspi.sib.core.exception.SIRollbackException
    * @throws com.ibm.websphere.sib.exception.SIResourceException
    * @throws com.ibm.wsspi.sib.core.exception.SIConnectionLostException
    * @throws com.ibm.websphere.sib.exception.SIErrorException
    */
   public synchronized void rollback()
      throws SIIncorrectCallException, 
             SIResourceException,
             SIConnectionLostException, 
             SIErrorException
   {
      if (tc.isEntryEnabled()) SibTr.entry(this, tc, "rollback");
      
      if (!valid) 
      {
         throw new SIIncorrectCallException(
            nls.getFormattedMessage("TRANSACTION_COMPLETE_SICO1022", null, null)
         );
      }
      
      // Mark this as invalid
      valid = false;

      CommsByteBuffer request = getCommsByteBuffer();
      request.putShort(getConnectionObjectID());
      request.putInt(getTransactionId());
      
      CommsByteBuffer reply = null;
      try
      {
         // Pass on call to server
         reply = jfapExchange(request,
                              JFapChannelConstants.SEG_ROLLBACK_TRANSACTION,
                              lowestPriority,
                              true);
         
         // Inform any associated consumers of the rollback, in case there are
         // redelivery ordering considerations. We do this regardless of the
         // success of the rollback, but need to do it after the rollback has
         // been sent to the ME.
         informConsumersOfRollback();
         
         short err = reply.getCommandCompletionCode(JFapChannelConstants.SEG_ROLLBACK_TRANSACTION_R);
         if (err != CommsConstants.SI_NO_EXCEPTION)
         {
            checkFor_SIIncorrectCallException(reply, err);
            checkFor_SIResourceException(reply, err);
            checkFor_SIConnectionLostException(reply, err);
            checkFor_SIErrorException(reply, err);
            defaultChecker(reply, err);
         }
      }
      catch (SIConnectionDroppedException e)
      {
         // No FFDC Code needed
         // PK60857:  The connection broke on us, but the ME side will eventually timeout
         // and assume rollback.  So we can safely consume this exception.  We previously
         // threw an exception here but that leaked a connection from the pool because
         // the tx never completed.
         if (tc.isDebugEnabled()) SibTr.debug(this, tc, "Connection failure during rollback.");
      }
      finally
      {
         if (reply != null) reply.release();
      }
      
      if (tc.isEntryEnabled()) SibTr.exit(this, tc, "rollback");
   }
   
   /**
    * This method will return true if the transaction has not been committed
    * or rolled back.
    * 
    * @return boolean 
    */
   public synchronized boolean isValid()
   {
      if (tc.isEntryEnabled()) SibTr.entry(this, tc, "isValid");
      if (tc.isEntryEnabled()) SibTr.exit(this, tc, "isValid", ""+valid);
      return valid;
   }
}
