/*******************************************************************************
 * Copyright (c) 2018 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.ssl.fat.pkcs12;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.security.AccessController;
import java.security.PrivilegedAction;

import org.junit.After;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestRule;

import com.ibm.websphere.simplicity.log.Log;
import com.ibm.ws.webcontainer.security.test.servlets.SSLBasicAuthClient;

import componenttest.annotation.AllowedFFDC;
import componenttest.topology.impl.LibertyServerFactory;
import componenttest.vulnerability.LeakedPasswordChecker;

public class DefaultPKCS12DoesNotExistSSLTest extends CommonSSLTest {
    private static final Class<?> c = DefaultPKCS12DoesNotExistSSLTest.class;
    private static boolean isOracle6 = false;

    public DefaultPKCS12DoesNotExistSSLTest() {
        super(LibertyServerFactory.getLibertyServer("com.ibm.ws.ssl.fat.pkcs12.nokeyp12"));
    }

    @Rule
    public TestRule passwordChecker = new LeakedPasswordChecker(server);

    @Override
    @Before
    public void setUp() throws Exception {
        super.setUp();
        if (!isIBMJVM() && isVersion6())
            isOracle6 = true;
    }

    @Override
    @After
    public void tearDown() throws Exception {
        super.tearDown();
        server.removeInstalledAppForValidation("basicauth");
    }

    /**
     * When the SSL feature is specified, a minimal SSL configuration is
     * specified in server.xml which includes the type,
     * <keyStore id="defaultKeyStore" type="PKCS12" password="liberty" />, and
     * a new key.12 file will be generated.
     *
     */
    @Test
    @AllowedFFDC("java.lang.NoClassDefFoundError")
    public void testDefaultMinimalSSLConfigWithPKCS12Type() throws Exception {

        Log.info(c, name.getMethodName(), "Entering " + name.getMethodName());
        String protocol = isOracle6 ? TLS_PROTOCOL : TLSV11_PROTOCOL;

        server.setServerConfigurationFile(DEFAULT_MINIMAL_SSL_CONFIG_WITH_PKCS12_TYPE);
        server.startServer(name.getMethodName() + ".log");

        // Requires info trace
        server.addInstalledAppForValidation("basicauth");

        assertNotNull("SSL TCP Channel did not start in time.",
                      server.waitForStringInLog("CWWKO0219I.*ssl"));
        assertNotNull("Need to wait for 'smarter planet' message (server is ready).",
                      server.waitForStringInLog("CWWKF0011I"));

        // Hit the servlet on the SSL port and validate it requires SSL
        SSLBasicAuthClient sslClient = null;
        String response = null;
        try {
            sslClient = createSSLClientWithTrust(DEFAULT_GENERATED_KEY_PATH, DEFAULT_GENERATED_KEY_PASSWORD, protocol);
            response = sslClient.accessUnprotectedServlet(SSLBasicAuthClient.UNPROTECTED_NO_SECURITY_CONSTRAINT);
        } catch (Exception e) {
        }
        if (response != null)
            assertTrue("Did not get the expected response",
                       sslClient.verifyUnauthenticatedResponse(response));
        else
            assertTrue("Did not get the expected response", false);

        // Verify the JKS file has not been generated

        assertFalse("The expected key file " + "resources/security/key.jks" + " was generated",
                    fileExists(server, "resources/security/key.jks"));

        // Verify key.jks has been generated
        assertTrue("The key file " + DEFAULT_GENERATED_KEY_PATH + " was generated",
                   fileExists(server, DEFAULT_GENERATED_KEY_PATH));

        if (server != null && server.isStarted()) {
            server.stopServer("CWNEN0050W", "SRVE0272W");
        }

        deleteFileIfExists(server, DEFAULT_GENERATED_KEY_PATH);
        deleteFileIfExists(server, ALTERNATE_GENERATED_KEY_PATH);

        Log.info(c, name.getMethodName(), "Exiting " + name.getMethodName());

    }

    private static boolean isIBMJVM() {
        String vendorName = AccessController.doPrivileged(new PrivilegedAction<String>() {
            @Override
            public String run() {
                return System.getProperty("java.vendor");
            }
        });
        return (vendorName != null && vendorName.toLowerCase().contains("ibm"));
    }

    private static boolean isVersion6() {
        String version = AccessController.doPrivileged(new PrivilegedAction<String>() {
            @Override
            public String run() {
                return System.getProperty("java.version");
            }
        });
        return (version != null && version.startsWith("1.6"));
    }

}
