/*******************************************************************************
 * Copyright (c) 2012 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package com.ibm.ws.fat.util;

import java.io.File;
import java.io.FileInputStream;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

import com.ibm.websphere.simplicity.Topology;

/**
 * <p>
 * Use this class to access the values of properties defined in
 * bootstrapping.properties, configuration.properties, properties.xml, and the
 * command line. Additional properties can also be set dynamically. Reading
 * properties through this object guarantees that the returned properties are
 * non-null.
 * </p><p>
 * The junit task from this Ant project should propagate system properties to the JVM through a local.properties file:<br>
 * /fattest.simplicity/autoFVT-defaults/src/ant/launch.xml
 * </p><p>
 * If that file changes, then instances of this class might return bizarre results.
 * </p><p>
 * This class also contains several statically defined property keys that can be
 * used to reference all of the known properties. Referencing properties using
 * the statically defined keys is good for three reasons:
 * <ul>
 * <li>makes it easy to locate all references to a specific property through Eclipse (good for debugging problems related to those properties).</li>
 * <li>You avoid syntax errors related to specifying invalid property keys in various places in your code. You also don't need to worry about updating property keys.</li>
 * <li>You only construct the String for each frequently-referenced property key once</li>
 * </ul>
 * </p>
 */
public class Props {

    private static final Logger LOG = Logger.getLogger(Props.class.getName());

    /*- Property keys defined by the command line -*/

    /*- Property keys from properties.xml -*/
    /**
     * The path to the root of this test component. Defined in properties.xml.
     */
    public static final String DIR_COMPONENT_ROOT = "dir.component.root";
    /**
     * The path to this test component's result directory, typically:<br>
     * <code>${component_Root_Directory}/results</code>.<br>
     * Defined in properties.xml.
     */
    public static final String DIR_LOG = "dir.log";

    /*- Property keys from TestBuild.xml and launch.xml -*/
    /**
     * The path to Moonstone's bootstrapping properties file. This file is
     * provided by the user (in a sandbox test environment), or by the Moonstone
     * infrastructure in an Aphid run or a Rapid Regression run. Defined in
     * launch.xml.
     */
    public static final String BOOTSTRAPPING_PROPERTIES = "bootstrapping.properties";
    /**
     * The path to Simplicity's configuration file. This file is generated by
     * launch.xml using configuration.properties. Defined in launch.xml.
     */
    public static final String SIMPLICITY_PROPERTIES = "simplicity.properties";
    /**
     * The path to this test bucket's configuration file. Defined in launch.xml.
     */
    public static final String CONFIGURATION_PROPERTIES = "configuration.properties";
    /**
     * The path to this test bucket's Java Logging properties file. This file is
     * used to control the of trace levels printed by this component (and
     * dependent APIs) during runtime. Defined and referenced in launch.xml.
     */
    public static final String LOGGING_PROPERTIES = "logging.properties";
    /**
     * The path to this test bucket's local properties file. This file is
     * generated by launch.xml, and contains all the properties found in
     * boostrapping.properties, configuration.properties, properties.xml, and
     * launch.xml. Defined in launch.xml.
     */
    public static final String LOCAL_PROPERTIES = "local.properties";

    /*- Property keys from configuration.properties -*/
    /**
     * Separator characters used to distinguish related blocks of output in test
     * bucket trace files. Small logging breaks are typically encapsulated by
     * larger logging breaks. Defined in configuration.properties.
     */
    public static final String LOGGING_BREAK_LARGE = "logging.break.large";
    /**
     * Separator characters used to distinguish related blocks of output in test
     * bucket trace files. Small logging breaks are typically encapsulated by
     * larger logging breaks. Defined in configuration.properties.
     */
    public static final String LOGGING_BREAK_MEDIUM = "logging.break.medium";
    /**
     * Separator characters used to distinguish related blocks of output in test
     * bucket trace files. Small logging breaks are typically encapsulated by
     * larger logging breaks. Defined in configuration.properties.
     */
    public static final String LOGGING_BREAK_SMALL = "logging.break.small";
    /**
     * The parent directory of all test case logs, relative to the dir.log directory specified by properties.xml
     */
    public static final String LOGGING_DIRS_PARENT = "logging.dirs.parent";
    /**
     * The prefix to use when creating directory names<br>
     * All characters must match: [a-zA-Z_0-9\\.]<br>
     * Specify the empty string to skip the prefix<br>
     * Recommended: Either "" or "Step"<br>
     * Only used if ${logging.dirs.enable}=true
     */
    public static final String LOGGING_DIRS_PREFIX = "logging.dirs.prefix";
    /**
     * The number of digits to use in the unique identifier for directory names<br>
     * Only used if ${logging.dirs.enable}=true
     */
    public static final String LOGGING_DIRS_DIGITS = "logging.dirs.digits";
    /**
     * true to include package information in each directory name;<br>
     * false to use simple class names only<br>
     * Only used if ${logging.dirs.enable}=true
     */
    public static final String LOGGING_DIRS_FULLNAME = "logging.dirs.fullname";

    /*- Property keys set in the local System (and redundantly Props.java) -*/
    /**
     * Java System property for the local JVM's file separator character.
     * Defined in the local System (and redundantly Props.java).
     */
    public static final String LOCAL_FILE_SEPARATOR = "file.separator";
    /**
     * Java System property for the local JVM's path separator character.
     * Defined in the local System (and redundantly Props.java).
     */
    public static final String LOCAL_PATH_SEPARATOR = "path.separator";
    /**
     * Java System property for the local JVM's line separator character.
     * Defined in the local System (and redundantly Props.java).
     */
    public static final String LOCAL_LINE_SEPARATOR = "line.separator";

    /*- Property keys set by Props.java -*/

    public static Props INSTANCE;

    /**
     * @return a global instance of this class that caches all properties from local.properties
     */
    public static Props getInstance() {
        if (INSTANCE == null) {
            setInstance(new Props());
        }
        return INSTANCE;
    }

    /**
     * Configures the global properties instance. Subclasses should call this method before any properties are read.
     * 
     * @param props the global properties instance to use
     */
    public static void setInstance(Props props) {
        INSTANCE = props;
    }

    /*- local variables -*/
    private final Properties properties = new Properties();
    private final String defaultValue = "";

    /**
     * Use the global instance exposed by {@link #getInstance()}
     */
    protected Props() {
        this.reloadProperties();
        this.printPropertyInformation();
    }

    /**
     * Load properties from local.properties (generated by launch.xml). Probably
     * only need to do this once during initialization.
     */
    public void reloadProperties() {
        if (LOG.isLoggable(Level.INFO)) {
            LOG.log(Level.INFO, "Initializing FAT Bucket Properties ...");
            LOG.log(Level.INFO, "System property used to locate the local properties file: " + LOCAL_PROPERTIES);
        }
        String propsFileName = System.getProperty(LOCAL_PROPERTIES);
        try {
            this.properties.load(new FileInputStream(propsFileName.trim()));
        } catch (Exception e) {
            if (LOG.isLoggable(Level.WARNING)) {
                LOG.log(Level.WARNING, "Unable to initialize properties", e);
            }
        }
    }

    /**
     * Log essential test bucket information regarding assumptions and input properties.
     */
    public void printPropertyInformation() {
        if (LOG.isLoggable(Level.INFO)) {
            LOG.log(Level.INFO, "Common Directories: ");
            LOG.log(Level.INFO, "Root Directory           - " + this.getFileProperty(DIR_COMPONENT_ROOT).getAbsolutePath());
            LOG.log(Level.INFO, "Result Directory         - " + this.getFileProperty(DIR_LOG).getAbsolutePath());
            LOG.log(Level.INFO, "User-input properties files: ");
            LOG.log(Level.INFO, "Bootstrapping Properties - " + this.getFileProperty(BOOTSTRAPPING_PROPERTIES).getAbsolutePath());
            LOG.log(Level.INFO, "Configuration Properties - " + this.getFileProperty(CONFIGURATION_PROPERTIES).getAbsolutePath());
            LOG.log(Level.INFO, "Logging Properties       - " + this.getFileProperty(LOGGING_PROPERTIES).getAbsolutePath());
            LOG.log(Level.INFO, "Generated properties files: ");
            LOG.log(Level.INFO, "Local Properties         - " + this.getFileProperty(LOCAL_PROPERTIES).getAbsolutePath());
            LOG.log(Level.INFO, "Simplicity Properties    - " + this.getFileProperty(SIMPLICITY_PROPERTIES).getAbsolutePath());
        }
    }

    /**
     * Retrieves a list of useful system properties that describe a JRE
     * 
     * @return a list of interesting Java property keys
     */
    protected String[] getJavaPropertyKeys() {
        return new String[] {
                             "os.name",
                             "os.version",
                             "os.arch",
                             "user.name",
                             "user.home",
                             "user.dir",
                             "java.home",
                             "java.version",
                             "java.fullversion",
                             "java.jcl.version",
                             "java.vendor",
                             "java.runtime.name",
                             "java.runtime.version",
                             "java.vm.name",
                             "java.vm.home",
                             "java.vm.version",
                             "java.vm.vendor",
                             "java.class.version",
                             "java.compiler",
                             "java.io.tmpdir" };
    }

    /**
     * Collects information about the current version of Java and other
     * dependencies.
     * 
     * @return a map of key/value pairs that describe the current environment
     */
    public PropertyMap collectVersionInformation() {
        PropertyMap properties = new PropertyMap();
        for (String prop : this.getJavaPropertyKeys()) {
            properties.put(prop, System.getProperty(prop));
        }
        properties.put("simplicity.version", Topology.SIMPLICITY_VERSION);
        //        properties.put("jiiws.version", SessionManager.getVersion());
        try {
            properties.put("host.address", InetAddress.getLocalHost().getHostAddress());
            properties.put("host.name", InetAddress.getLocalHost().getHostName());
            properties.put("host.name.canonical", InetAddress.getLocalHost().getCanonicalHostName());
        } catch (UnknownHostException e) {
            // it doesn't really matter; just don't print these properties
        }
        return properties;
    }

    /**
     * Determines whether or not a property is set.
     * 
     * @param prop
     *            The key of the property you want to check for the existence of
     * @return true if the input property is set
     */
    public boolean isSet(String prop) {
        if (prop == null) {
            return false;
        }
        String value = this.properties.getProperty(prop);
        return (value != null);
    }

    /**
     * Determines whether a property is set to a String composed of white-space
     * characters.
     * 
     * @param prop
     *            The key of the property you want to check
     * @return true if the input property is set, but either contains spaces or
     *         is the empty String
     */
    public boolean isSetToBlank(String prop) {
        if (prop == null) {
            return false;
        }
        String value = this.properties.getProperty(prop);
        if (value == null) {
            return false;
        }
        return (this.defaultValue.equals(value.trim()));
    }

    /**
     * Determines whether a property is set to a String composed of
     * non-white-space characters.
     * 
     * @param prop
     *            The key of the property you want to check
     * @return true if the input property is set and is not blank
     */
    public boolean isSetToText(String prop) {
        if (prop == null) {
            return false;
        }
        String value = this.properties.getProperty(prop);
        if (value == null) {
            return false;
        }
        return !(this.defaultValue.equals(value.trim()));
    }

    /**
     * Determine the value of a property as a String
     * 
     * @param prop
     *            The key of the property you want to check
     * @return The value of the input property
     */
    public String getProperty(String prop) {
        if (prop == null) {
            return this.defaultValue;
        }
        return this.properties.getProperty(prop, this.defaultValue).trim();
    }

    /**
     * Determine the value of a property as an int.
     * 
     * @param prop
     *            The key of the property you want to check
     * @return The value of the input property, or zero if the value of the
     *         property could not be parsed
     */
    public int getIntProperty(String prop) {
        String value = getProperty(prop);
        try {
            return Integer.parseInt(value);
        } catch (Exception e) {
            if (LOG.isLoggable(Level.FINE)) {
                LOG.log(Level.FINE, "Encountered problems loading property: '" + prop + "'.  Property found: '" + value + "'.  Using default value of 0 instead", e);
            }
            return 0;
        }
    }

    /**
     * Determine the value of a property as a boolean. The <code>boolean</code>
     * returned represents the value <code>true</code> if the value of the
     * property is not <code>null</code> and is equal, ignoring case, to the
     * string <code>"true"</code>.<br>
     * <br>
     * Example: <tt>Boolean.valueOf("True")</tt> returns <tt>true</tt>.<br>
     * Example: <tt>Boolean.valueOf("yes")</tt> returns <tt>false</tt>.
     * 
     * @param prop
     *            The key of the property you want to check
     * @return The value of the input property as a boolean
     */
    public boolean getBooleanProperty(String prop) {
        return Boolean.valueOf(getProperty(prop)).booleanValue();
    }

    /**
     * Determine the value of a property as a File. Creates a new
     * <code>File</code> instance by converting the value specified by the
     * property key into an abstract pathname. If the given string is the empty
     * string, then the result is the empty abstract pathname.
     * 
     * @param prop
     *            The key of the property you want to check
     * @return The value of the input property as a File
     */
    public File getFileProperty(String prop) {
        return new File(getProperty(prop));
    }

    /**
     * Sets a property if the input key and value are not null. If either
     * parameter is null, no property is set.
     * 
     * @param prop
     *            The key of the property you want to set
     * @param value
     *            The value of the property you want to set
     */
    public void setProperty(String prop, String value) {
        if (prop != null && value != null) {
            this.properties.setProperty(prop, value);
        }
    }

    /**
     * Sets an int property if the input key is not null. If the key is null, no
     * property is set.
     * 
     * @param prop
     *            The key of the property you want to set
     * @param value
     *            The value of the property you want to set
     */
    public void setIntProperty(String prop, int value) {
        setProperty(prop, new Integer(value).toString());
    }

    /**
     * Sets a boolean property if the input key is not null. If the key is null,
     * no property is set.
     * 
     * @param prop
     *            The key of the property you want to set
     * @param value
     *            The value of the property you want to set
     */
    public void setBooleanProperty(String prop, boolean value) {
        setProperty(prop, new Boolean(value).toString());
    }

    /**
     * Sets a File property if the input key and value are not null. If either
     * parameter is null, no property is set.
     * 
     * @param prop
     *            The key of the property you want to set
     * @param file
     *            The value of the property you want to set
     */
    public void setFileProperty(String prop, File file) {
        if (file != null) {
            try {
                setProperty(prop, file.getCanonicalPath());
            } catch (Exception e) {
                setProperty(prop, file.getAbsolutePath());
            }
        }
        // else, if the file is null, there's no property to set!
    }

}
